/*
* Copyright (C) 2016 Texas Instruments Incorporated - www.ti.com
*
*
*  Redistribution and use in source and binary forms, with or without
*  modification, are permitted provided that the following conditions
*  are met:
*
*    Redistributions of source code must retain the above copyright
*    notice, this list of conditions and the following disclaimer.
*
*    Redistributions in binary form must reproduce the above copyright
*    notice, this list of conditions and the following disclaimer in the
*    documentation and/or other materials provided with the
*    distribution.
*
*    Neither the name of Texas Instruments Incorporated nor the names of
*    its contributors may be used to endorse or promote products derived
*    from this software without specific prior written permission.
*
*  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
*  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
*  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
*  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
*  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
*  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
*  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
*  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
*  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
*  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
*  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
*/

/* HALCoGen Register Definition Includes Files */
#include "sys_common.h"
#include "sys_dma.h"
#include "sys_pmu.h"

/* HALCoGen Drivers Include Files */
#include "crc.h"

/* CG Tools Include Files */
#include <crc_tbl.h>
#include <stdio.h>

/* Application Include Files */
#include "CRC64_calc.h"


extern const CRC_TABLE _my_crc_table;
volatile uint32 cycles = 0ul;

#ifndef crcREG
#define crcREG crcREG1
/* Note: This is just a helper to identify RM57 and LC43 devices, as they need different handling for the DMA */
#define RM57_OR_LC43
#else
#undef crcREG1
#undef crcREG2
#undef RM57_OR_LC43
#endif

volatile uint64 u64Signature  = 0ull;

int main(void)
{
	uint32          i;

	_pmuInit_();
	_pmuEnableCountersGlobal_();

	crcInit();

	if(1)
	{
		printf("Start CRC Calculation in SW Mode\n");

		_pmuResetCycleCounter_();
		_pmuStartCounters_(pmuCYCLE_COUNTER);

		for (i = 0ul ; i < _my_crc_table.num_recs ; i++)
		{
			//uint64   u64Signature = 0ull;
			uint64 * pu64Address  = (void *)(-1); /* NULL is a valid address */
			uint32   u32Size      = 0ul;

			/* Check Alignment of Start Address */
			if (0ull == (_my_crc_table.recs[i].addr % 8ull))
			{
				pu64Address = (uint64*)(_my_crc_table.recs[i].addr);

				/* Check Alignment of Size */
				if (0ul == (_my_crc_table.recs[i].size % 8ul))
				{
					/* Assign and adjust size to increments of 8 Bytes */
					u32Size = _my_crc_table.recs[i].size / 8ul;

					/* Compute CRC in SW */
					u64Signature = calc_crc64(pu64Address, u32Size);

					/* Check CRC */
					if (u64Signature != _my_crc_table.recs[i].crc_value)
					{
						/* Fail */

						printf("Wrong CRC in SW Module @ address: 0x%08X\n", _my_crc_table.recs[i].addr);
						while(1);
					}
					else
					{
						/* Pass */
					}
				}
				else
				{
					/* Wrong Alignment of Size */
				}
			}
			else
			{
				/* Wrong Alignment of Start Address */
			}
		}

		cycles = _pmuGetCycleCount_();
		_pmuStopCounters_(pmuCYCLE_COUNTER);
		printf("Finished CRC calculation, CPU cycles: %d\n", cycles);
	}

	if(1)
	{
		crcConfig_t     sCrcParams;

		printf("Start CRC Calculation in Full CPU Mode\n");

		sCrcParams.crc_channel   = CRC_CH1;
		sCrcParams.mode          = CRC_FULL_CPU;
		sCrcParams.pcount        = 0u; /* All counters are disabled in Full CPU mode */
		sCrcParams.scount        = 0u; /* All counters are disabled in Full CPU mode */
		sCrcParams.wdg_preload   = 0u; /* All counters are disabled in Full CPU mode */
		sCrcParams.block_preload = 0u; /* All counters are disabled in Full CPU mode */

		crcSetConfig(crcREG, &sCrcParams);

		_pmuResetCycleCounter_();
		_pmuStartCounters_(pmuCYCLE_COUNTER);

		for (i = 0ul ; i < _my_crc_table.num_recs ; i++)
		{
			crcModConfig_t  sCrcModConfig;
			//volatile uint64 u64Signature  = 0ull;

			sCrcModConfig.crc_channel  = CRC_CH1;
			sCrcModConfig.mode         = CRC_FULL_CPU;
			sCrcModConfig.data_length  = (_my_crc_table.recs[i].size + 7ul) / 8ul;
			sCrcModConfig.src_data_pat = (uint64*)_my_crc_table.recs[i].addr;

			/* Compute CRC with HW Module, Full CPU Mode */
			crcChannelReset(crcREG, CRC_CH1);
			crcSignGen(crcREG, &sCrcModConfig);
			u64Signature = crcGetPSASig(crcREG, CRC_CH1);

			/* Check CRC */
			if (u64Signature != _my_crc_table.recs[i].crc_value)
			{
				printf("Wrong CRC in SW Module @ address: 0x%08X\n", _my_crc_table.recs[i].addr);
				while(1); /* Fail */
			}
			else
			{
				/* Pass */
			}
		}

		cycles = _pmuGetCycleCount_();
		_pmuStopCounters_(pmuCYCLE_COUNTER);
		printf("Finished CRC calculation, CPU cycles: %d\n", cycles);
	}



	if(1)
	{
		printf("Start CRC Calculation in Semi CPU Mode (Polling)\n");

		/* Enable DMA module : this brings DMA out of reset */
		dmaEnable();

		/* Enable all interrupts for Channel 0 */
		crcEnableNotification(crcREG,
				CRC_CH1_CC |
				CRC_CH1_FAIL |
				CRC_CH1_OR |
				CRC_CH1_UR |
				CRC_CH1_TO);

		_pmuResetCycleCounter_();
		_pmuStartCounters_(pmuCYCLE_COUNTER);

		for (i = 0ul ; i < _my_crc_table.num_recs ; i++)
		{
			crcConfig_t     sCrcParams;
			g_dmaCTRL       g_dmaCTRLPKT;
			//volatile uint64 u64Signature  = 0ull;

			/* Configure DMA Control Packed (structure is part of dma.c) */
			g_dmaCTRLPKT.SADD      = _my_crc_table.recs[i].addr;  /* initial source address */
			g_dmaCTRLPKT.DADD      = (uint32_t)(&(crcREG->PSA_SIGREGL1)); /* initial destination address */
			g_dmaCTRLPKT.CHCTRL    = 0ul; /* channel control */
			g_dmaCTRLPKT.RDSIZE    = ACCESS_64_BIT; /* read size */
			g_dmaCTRLPKT.WRSIZE    = ACCESS_64_BIT; /* write size */
			g_dmaCTRLPKT.FRCNT     = 1; /* frame count */
			g_dmaCTRLPKT.ELCNT     = (_my_crc_table.recs[i].size + 7ul) / 8ul;   /* element count */
			g_dmaCTRLPKT.ELSOFFSET = 0ul << g_dmaCTRLPKT.RDSIZE; /* element source offset */
			g_dmaCTRLPKT.FRSOFFSET = 0ul << g_dmaCTRLPKT.RDSIZE; /* frame source offset */
			g_dmaCTRLPKT.ELDOFFSET = 0ul << g_dmaCTRLPKT.WRSIZE; /* element destination offset */
			g_dmaCTRLPKT.FRDOFFSET = 0ul << g_dmaCTRLPKT.WRSIZE; /* frame destination offset */
#ifdef RM57_OR_LC43
			g_dmaCTRLPKT.PORTASGN  = 1ul; /* Port A Read (Flash) - Port B Write (CRC) */
#else
			g_dmaCTRLPKT.PORTASGN  = 4ul; /* only Port B */
#endif
			g_dmaCTRLPKT.TTYPE     = FRAME_TRANSFER ; /* transfer type */
			g_dmaCTRLPKT.ADDMODERD = ADDR_INC1;     /* address mode read */
			g_dmaCTRLPKT.ADDMODEWR = ADDR_FIXED;     /* address mode write */
			g_dmaCTRLPKT.AUTOINIT  = AUTOINIT_OFF;     /* autoinit off */

			sCrcParams.crc_channel   = CRC_CH1;
			sCrcParams.mode          = CRC_SEMI_CPU;
			sCrcParams.pcount        = g_dmaCTRLPKT.ELCNT;
			sCrcParams.scount        = 1u;
			sCrcParams.wdg_preload   = 0u;
			sCrcParams.block_preload = 0u;

			crcChannelReset(crcREG, CRC_CH1);
			crcSetConfig(crcREG, &sCrcParams);

			//printf("Checking CRC for address 0x%08X, %d elements\r\n", g_dmaCTRLPKT.SADD, g_dmaCTRLPKT.ELCNT);

			/* Assign DMA Control Packet to Channel 0 */
			dmaSetCtrlPacket(DMA_CH0, g_dmaCTRLPKT);

			/* Trigger DMA Channel 0 s/w request */
			dmaSetChEnable(DMA_CH0, DMA_SW);

			while( (dmaREG->SWCHENAS & 1ul) | (crcREG->BUSY & 1ul) )
			{
				/* Wait until DMA finsihed and CRC indicates block compressed */
			}

			u64Signature = crcGetSectorSig(crcREG, CRC_CH1);

			/* Clear CH1_CCIT bit */
			crcREG->STATUS = CRC_CH1_CC;

			/* Check CRC */
			if (u64Signature != _my_crc_table.recs[i].crc_value)
			{
				printf("Wrong CRC in SW Module @ address: 0x%08X\n", _my_crc_table.recs[i].addr);
			}
			else
			{
				/* Pass */
			}
		}

		cycles = _pmuGetCycleCount_();
		_pmuStopCounters_(pmuCYCLE_COUNTER);
		printf("Finished CRC calculation, CPU cycles: %d\n", cycles);
	}

	while(1); /* Loop for ever. */
	
	return 0;
}
